try {
  self["workbox:core:6.4.1"] && _();
} catch (e) {
}
const fallback = (code, ...args) => {
  let msg = code;
  if (args.length > 0) {
    msg += ` :: ${JSON.stringify(args)}`;
  }
  return msg;
};
const messageGenerator = fallback;
class WorkboxError extends Error {
  constructor(errorCode, details) {
    const message = messageGenerator(errorCode, details);
    super(message);
    this.name = errorCode;
    this.details = details;
  }
}
const _cacheNameDetails = {
  googleAnalytics: "googleAnalytics",
  precache: "precache-v2",
  prefix: "workbox",
  runtime: "runtime",
  suffix: typeof registration !== "undefined" ? registration.scope : ""
};
const _createCacheName = (cacheName) => {
  return [_cacheNameDetails.prefix, cacheName, _cacheNameDetails.suffix].filter((value) => value && value.length > 0).join("-");
};
const eachCacheNameDetail = (fn) => {
  for (const key of Object.keys(_cacheNameDetails)) {
    fn(key);
  }
};
const cacheNames = {
  updateDetails: (details) => {
    eachCacheNameDetail((key) => {
      if (typeof details[key] === "string") {
        _cacheNameDetails[key] = details[key];
      }
    });
  },
  getGoogleAnalyticsName: (userCacheName) => {
    return userCacheName || _createCacheName(_cacheNameDetails.googleAnalytics);
  },
  getPrecacheName: (userCacheName) => {
    return userCacheName || _createCacheName(_cacheNameDetails.precache);
  },
  getPrefix: () => {
    return _cacheNameDetails.prefix;
  },
  getRuntimeName: (userCacheName) => {
    return userCacheName || _createCacheName(_cacheNameDetails.runtime);
  },
  getSuffix: () => {
    return _cacheNameDetails.suffix;
  }
};
const logger = null;
function waitUntil(event, asyncFn) {
  const returnPromise = asyncFn();
  event.waitUntil(returnPromise);
  return returnPromise;
}
try {
  self["workbox:precaching:6.4.1"] && _();
} catch (e) {
}
const REVISION_SEARCH_PARAM = "__WB_REVISION__";
function createCacheKey(entry) {
  if (!entry) {
    throw new WorkboxError("add-to-cache-list-unexpected-type", { entry });
  }
  if (typeof entry === "string") {
    const urlObject = new URL(entry, location.href);
    return {
      cacheKey: urlObject.href,
      url: urlObject.href
    };
  }
  const { revision, url } = entry;
  if (!url) {
    throw new WorkboxError("add-to-cache-list-unexpected-type", { entry });
  }
  if (!revision) {
    const urlObject = new URL(url, location.href);
    return {
      cacheKey: urlObject.href,
      url: urlObject.href
    };
  }
  const cacheKeyURL = new URL(url, location.href);
  const originalURL = new URL(url, location.href);
  cacheKeyURL.searchParams.set(REVISION_SEARCH_PARAM, revision);
  return {
    cacheKey: cacheKeyURL.href,
    url: originalURL.href
  };
}
class PrecacheInstallReportPlugin {
  constructor() {
    this.updatedURLs = [];
    this.notUpdatedURLs = [];
    this.handlerWillStart = async ({ request, state }) => {
      if (state) {
        state.originalRequest = request;
      }
    };
    this.cachedResponseWillBeUsed = async ({ event, state, cachedResponse }) => {
      if (event.type === "install") {
        if (state && state.originalRequest && state.originalRequest instanceof Request) {
          const url = state.originalRequest.url;
          if (cachedResponse) {
            this.notUpdatedURLs.push(url);
          } else {
            this.updatedURLs.push(url);
          }
        }
      }
      return cachedResponse;
    };
  }
}
class PrecacheCacheKeyPlugin {
  constructor({ precacheController: precacheController2 }) {
    this.cacheKeyWillBeUsed = async ({ request, params }) => {
      const cacheKey = (params === null || params === void 0 ? void 0 : params.cacheKey) || this._precacheController.getCacheKeyForURL(request.url);
      return cacheKey ? new Request(cacheKey, { headers: request.headers }) : request;
    };
    this._precacheController = precacheController2;
  }
}
let supportStatus;
function canConstructResponseFromBodyStream() {
  if (supportStatus === void 0) {
    const testResponse = new Response("");
    if ("body" in testResponse) {
      try {
        new Response(testResponse.body);
        supportStatus = true;
      } catch (error) {
        supportStatus = false;
      }
    }
    supportStatus = false;
  }
  return supportStatus;
}
async function copyResponse(response, modifier) {
  let origin = null;
  if (response.url) {
    const responseURL = new URL(response.url);
    origin = responseURL.origin;
  }
  if (origin !== self.location.origin) {
    throw new WorkboxError("cross-origin-copy-response", { origin });
  }
  const clonedResponse = response.clone();
  const responseInit = {
    headers: new Headers(clonedResponse.headers),
    status: clonedResponse.status,
    statusText: clonedResponse.statusText
  };
  const modifiedResponseInit = modifier ? modifier(responseInit) : responseInit;
  const body = canConstructResponseFromBodyStream() ? clonedResponse.body : await clonedResponse.blob();
  return new Response(body, modifiedResponseInit);
}
const getFriendlyURL = (url) => {
  const urlObj = new URL(String(url), location.href);
  return urlObj.href.replace(new RegExp(`^${location.origin}`), "");
};
function stripParams(fullURL, ignoreParams) {
  const strippedURL = new URL(fullURL);
  for (const param of ignoreParams) {
    strippedURL.searchParams.delete(param);
  }
  return strippedURL.href;
}
async function cacheMatchIgnoreParams(cache, request, ignoreParams, matchOptions) {
  const strippedRequestURL = stripParams(request.url, ignoreParams);
  if (request.url === strippedRequestURL) {
    return cache.match(request, matchOptions);
  }
  const keysOptions = Object.assign(Object.assign({}, matchOptions), { ignoreSearch: true });
  const cacheKeys = await cache.keys(request, keysOptions);
  for (const cacheKey of cacheKeys) {
    const strippedCacheKeyURL = stripParams(cacheKey.url, ignoreParams);
    if (strippedRequestURL === strippedCacheKeyURL) {
      return cache.match(cacheKey, matchOptions);
    }
  }
  return;
}
class Deferred {
  constructor() {
    this.promise = new Promise((resolve, reject) => {
      this.resolve = resolve;
      this.reject = reject;
    });
  }
}
const quotaErrorCallbacks = new Set();
async function executeQuotaErrorCallbacks() {
  for (const callback of quotaErrorCallbacks) {
    await callback();
  }
}
function timeout(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}
try {
  self["workbox:strategies:6.4.1"] && _();
} catch (e) {
}
function toRequest(input) {
  return typeof input === "string" ? new Request(input) : input;
}
class StrategyHandler {
  constructor(strategy, options) {
    this._cacheKeys = {};
    Object.assign(this, options);
    this.event = options.event;
    this._strategy = strategy;
    this._handlerDeferred = new Deferred();
    this._extendLifetimePromises = [];
    this._plugins = [...strategy.plugins];
    this._pluginStateMap = new Map();
    for (const plugin of this._plugins) {
      this._pluginStateMap.set(plugin, {});
    }
    this.event.waitUntil(this._handlerDeferred.promise);
  }
  async fetch(input) {
    const { event } = this;
    let request = toRequest(input);
    if (request.mode === "navigate" && event instanceof FetchEvent && event.preloadResponse) {
      const possiblePreloadResponse = await event.preloadResponse;
      if (possiblePreloadResponse) {
        return possiblePreloadResponse;
      }
    }
    const originalRequest = this.hasCallback("fetchDidFail") ? request.clone() : null;
    try {
      for (const cb of this.iterateCallbacks("requestWillFetch")) {
        request = await cb({ request: request.clone(), event });
      }
    } catch (err) {
      if (err instanceof Error) {
        throw new WorkboxError("plugin-error-request-will-fetch", {
          thrownErrorMessage: err.message
        });
      }
    }
    const pluginFilteredRequest = request.clone();
    try {
      let fetchResponse;
      fetchResponse = await fetch(request, request.mode === "navigate" ? void 0 : this._strategy.fetchOptions);
      if (false)
        ;
      for (const callback of this.iterateCallbacks("fetchDidSucceed")) {
        fetchResponse = await callback({
          event,
          request: pluginFilteredRequest,
          response: fetchResponse
        });
      }
      return fetchResponse;
    } catch (error) {
      if (originalRequest) {
        await this.runCallbacks("fetchDidFail", {
          error,
          event,
          originalRequest: originalRequest.clone(),
          request: pluginFilteredRequest.clone()
        });
      }
      throw error;
    }
  }
  async fetchAndCachePut(input) {
    const response = await this.fetch(input);
    const responseClone = response.clone();
    void this.waitUntil(this.cachePut(input, responseClone));
    return response;
  }
  async cacheMatch(key) {
    const request = toRequest(key);
    let cachedResponse;
    const { cacheName, matchOptions } = this._strategy;
    const effectiveRequest = await this.getCacheKey(request, "read");
    const multiMatchOptions = Object.assign(Object.assign({}, matchOptions), { cacheName });
    cachedResponse = await caches.match(effectiveRequest, multiMatchOptions);
    for (const callback of this.iterateCallbacks("cachedResponseWillBeUsed")) {
      cachedResponse = await callback({
        cacheName,
        matchOptions,
        cachedResponse,
        request: effectiveRequest,
        event: this.event
      }) || void 0;
    }
    return cachedResponse;
  }
  async cachePut(key, response) {
    const request = toRequest(key);
    await timeout(0);
    const effectiveRequest = await this.getCacheKey(request, "write");
    if (!response) {
      throw new WorkboxError("cache-put-with-no-response", {
        url: getFriendlyURL(effectiveRequest.url)
      });
    }
    const responseToCache = await this._ensureResponseSafeToCache(response);
    if (!responseToCache) {
      return false;
    }
    const { cacheName, matchOptions } = this._strategy;
    const cache = await self.caches.open(cacheName);
    const hasCacheUpdateCallback = this.hasCallback("cacheDidUpdate");
    const oldResponse = hasCacheUpdateCallback ? await cacheMatchIgnoreParams(cache, effectiveRequest.clone(), ["__WB_REVISION__"], matchOptions) : null;
    try {
      await cache.put(effectiveRequest, hasCacheUpdateCallback ? responseToCache.clone() : responseToCache);
    } catch (error) {
      if (error instanceof Error) {
        if (error.name === "QuotaExceededError") {
          await executeQuotaErrorCallbacks();
        }
        throw error;
      }
    }
    for (const callback of this.iterateCallbacks("cacheDidUpdate")) {
      await callback({
        cacheName,
        oldResponse,
        newResponse: responseToCache.clone(),
        request: effectiveRequest,
        event: this.event
      });
    }
    return true;
  }
  async getCacheKey(request, mode) {
    const key = `${request.url} | ${mode}`;
    if (!this._cacheKeys[key]) {
      let effectiveRequest = request;
      for (const callback of this.iterateCallbacks("cacheKeyWillBeUsed")) {
        effectiveRequest = toRequest(await callback({
          mode,
          request: effectiveRequest,
          event: this.event,
          params: this.params
        }));
      }
      this._cacheKeys[key] = effectiveRequest;
    }
    return this._cacheKeys[key];
  }
  hasCallback(name) {
    for (const plugin of this._strategy.plugins) {
      if (name in plugin) {
        return true;
      }
    }
    return false;
  }
  async runCallbacks(name, param) {
    for (const callback of this.iterateCallbacks(name)) {
      await callback(param);
    }
  }
  *iterateCallbacks(name) {
    for (const plugin of this._strategy.plugins) {
      if (typeof plugin[name] === "function") {
        const state = this._pluginStateMap.get(plugin);
        const statefulCallback = (param) => {
          const statefulParam = Object.assign(Object.assign({}, param), { state });
          return plugin[name](statefulParam);
        };
        yield statefulCallback;
      }
    }
  }
  waitUntil(promise) {
    this._extendLifetimePromises.push(promise);
    return promise;
  }
  async doneWaiting() {
    let promise;
    while (promise = this._extendLifetimePromises.shift()) {
      await promise;
    }
  }
  destroy() {
    this._handlerDeferred.resolve(null);
  }
  async _ensureResponseSafeToCache(response) {
    let responseToCache = response;
    let pluginsUsed = false;
    for (const callback of this.iterateCallbacks("cacheWillUpdate")) {
      responseToCache = await callback({
        request: this.request,
        response: responseToCache,
        event: this.event
      }) || void 0;
      pluginsUsed = true;
      if (!responseToCache) {
        break;
      }
    }
    if (!pluginsUsed) {
      if (responseToCache && responseToCache.status !== 200) {
        responseToCache = void 0;
      }
    }
    return responseToCache;
  }
}
class Strategy {
  constructor(options = {}) {
    this.cacheName = cacheNames.getRuntimeName(options.cacheName);
    this.plugins = options.plugins || [];
    this.fetchOptions = options.fetchOptions;
    this.matchOptions = options.matchOptions;
  }
  handle(options) {
    const [responseDone] = this.handleAll(options);
    return responseDone;
  }
  handleAll(options) {
    if (options instanceof FetchEvent) {
      options = {
        event: options,
        request: options.request
      };
    }
    const event = options.event;
    const request = typeof options.request === "string" ? new Request(options.request) : options.request;
    const params = "params" in options ? options.params : void 0;
    const handler = new StrategyHandler(this, { event, request, params });
    const responseDone = this._getResponse(handler, request, event);
    const handlerDone = this._awaitComplete(responseDone, handler, request, event);
    return [responseDone, handlerDone];
  }
  async _getResponse(handler, request, event) {
    await handler.runCallbacks("handlerWillStart", { event, request });
    let response = void 0;
    try {
      response = await this._handle(request, handler);
      if (!response || response.type === "error") {
        throw new WorkboxError("no-response", { url: request.url });
      }
    } catch (error) {
      if (error instanceof Error) {
        for (const callback of handler.iterateCallbacks("handlerDidError")) {
          response = await callback({ error, event, request });
          if (response) {
            break;
          }
        }
      }
      if (!response) {
        throw error;
      }
    }
    for (const callback of handler.iterateCallbacks("handlerWillRespond")) {
      response = await callback({ event, request, response });
    }
    return response;
  }
  async _awaitComplete(responseDone, handler, request, event) {
    let response;
    let error;
    try {
      response = await responseDone;
    } catch (error2) {
    }
    try {
      await handler.runCallbacks("handlerDidRespond", {
        event,
        request,
        response
      });
      await handler.doneWaiting();
    } catch (waitUntilError) {
      if (waitUntilError instanceof Error) {
        error = waitUntilError;
      }
    }
    await handler.runCallbacks("handlerDidComplete", {
      event,
      request,
      response,
      error
    });
    handler.destroy();
    if (error) {
      throw error;
    }
  }
}
class PrecacheStrategy extends Strategy {
  constructor(options = {}) {
    options.cacheName = cacheNames.getPrecacheName(options.cacheName);
    super(options);
    this._fallbackToNetwork = options.fallbackToNetwork === false ? false : true;
    this.plugins.push(PrecacheStrategy.copyRedirectedCacheableResponsesPlugin);
  }
  async _handle(request, handler) {
    const response = await handler.cacheMatch(request);
    if (response) {
      return response;
    }
    if (handler.event && handler.event.type === "install") {
      return await this._handleInstall(request, handler);
    }
    return await this._handleFetch(request, handler);
  }
  async _handleFetch(request, handler) {
    let response;
    const params = handler.params || {};
    if (this._fallbackToNetwork) {
      const integrityInManifest = params.integrity;
      const integrityInRequest = request.integrity;
      const noIntegrityConflict = !integrityInRequest || integrityInRequest === integrityInManifest;
      response = await handler.fetch(new Request(request, {
        integrity: integrityInRequest || integrityInManifest
      }));
      if (integrityInManifest && noIntegrityConflict) {
        this._useDefaultCacheabilityPluginIfNeeded();
        await handler.cachePut(request, response.clone());
      }
    } else {
      throw new WorkboxError("missing-precache-entry", {
        cacheName: this.cacheName,
        url: request.url
      });
    }
    return response;
  }
  async _handleInstall(request, handler) {
    this._useDefaultCacheabilityPluginIfNeeded();
    const response = await handler.fetch(request);
    const wasCached = await handler.cachePut(request, response.clone());
    if (!wasCached) {
      throw new WorkboxError("bad-precaching-response", {
        url: request.url,
        status: response.status
      });
    }
    return response;
  }
  _useDefaultCacheabilityPluginIfNeeded() {
    let defaultPluginIndex = null;
    let cacheWillUpdatePluginCount = 0;
    for (const [index, plugin] of this.plugins.entries()) {
      if (plugin === PrecacheStrategy.copyRedirectedCacheableResponsesPlugin) {
        continue;
      }
      if (plugin === PrecacheStrategy.defaultPrecacheCacheabilityPlugin) {
        defaultPluginIndex = index;
      }
      if (plugin.cacheWillUpdate) {
        cacheWillUpdatePluginCount++;
      }
    }
    if (cacheWillUpdatePluginCount === 0) {
      this.plugins.push(PrecacheStrategy.defaultPrecacheCacheabilityPlugin);
    } else if (cacheWillUpdatePluginCount > 1 && defaultPluginIndex !== null) {
      this.plugins.splice(defaultPluginIndex, 1);
    }
  }
}
PrecacheStrategy.defaultPrecacheCacheabilityPlugin = {
  async cacheWillUpdate({ response }) {
    if (!response || response.status >= 400) {
      return null;
    }
    return response;
  }
};
PrecacheStrategy.copyRedirectedCacheableResponsesPlugin = {
  async cacheWillUpdate({ response }) {
    return response.redirected ? await copyResponse(response) : response;
  }
};
class PrecacheController {
  constructor({ cacheName, plugins = [], fallbackToNetwork = true } = {}) {
    this._urlsToCacheKeys = new Map();
    this._urlsToCacheModes = new Map();
    this._cacheKeysToIntegrities = new Map();
    this._strategy = new PrecacheStrategy({
      cacheName: cacheNames.getPrecacheName(cacheName),
      plugins: [
        ...plugins,
        new PrecacheCacheKeyPlugin({ precacheController: this })
      ],
      fallbackToNetwork
    });
    this.install = this.install.bind(this);
    this.activate = this.activate.bind(this);
  }
  get strategy() {
    return this._strategy;
  }
  precache(entries) {
    this.addToCacheList(entries);
    if (!this._installAndActiveListenersAdded) {
      self.addEventListener("install", this.install);
      self.addEventListener("activate", this.activate);
      this._installAndActiveListenersAdded = true;
    }
  }
  addToCacheList(entries) {
    const urlsToWarnAbout = [];
    for (const entry of entries) {
      if (typeof entry === "string") {
        urlsToWarnAbout.push(entry);
      } else if (entry && entry.revision === void 0) {
        urlsToWarnAbout.push(entry.url);
      }
      const { cacheKey, url } = createCacheKey(entry);
      const cacheMode = typeof entry !== "string" && entry.revision ? "reload" : "default";
      if (this._urlsToCacheKeys.has(url) && this._urlsToCacheKeys.get(url) !== cacheKey) {
        throw new WorkboxError("add-to-cache-list-conflicting-entries", {
          firstEntry: this._urlsToCacheKeys.get(url),
          secondEntry: cacheKey
        });
      }
      if (typeof entry !== "string" && entry.integrity) {
        if (this._cacheKeysToIntegrities.has(cacheKey) && this._cacheKeysToIntegrities.get(cacheKey) !== entry.integrity) {
          throw new WorkboxError("add-to-cache-list-conflicting-integrities", {
            url
          });
        }
        this._cacheKeysToIntegrities.set(cacheKey, entry.integrity);
      }
      this._urlsToCacheKeys.set(url, cacheKey);
      this._urlsToCacheModes.set(url, cacheMode);
      if (urlsToWarnAbout.length > 0) {
        const warningMessage = `Workbox is precaching URLs without revision info: ${urlsToWarnAbout.join(", ")}
This is generally NOT safe. Learn more at https://bit.ly/wb-precache`;
        {
          console.warn(warningMessage);
        }
      }
    }
  }
  install(event) {
    return waitUntil(event, async () => {
      const installReportPlugin = new PrecacheInstallReportPlugin();
      this.strategy.plugins.push(installReportPlugin);
      for (const [url, cacheKey] of this._urlsToCacheKeys) {
        const integrity = this._cacheKeysToIntegrities.get(cacheKey);
        const cacheMode = this._urlsToCacheModes.get(url);
        const request = new Request(url, {
          integrity,
          cache: cacheMode,
          credentials: "same-origin"
        });
        await Promise.all(this.strategy.handleAll({
          params: { cacheKey },
          request,
          event
        }));
      }
      const { updatedURLs, notUpdatedURLs } = installReportPlugin;
      return { updatedURLs, notUpdatedURLs };
    });
  }
  activate(event) {
    return waitUntil(event, async () => {
      const cache = await self.caches.open(this.strategy.cacheName);
      const currentlyCachedRequests = await cache.keys();
      const expectedCacheKeys = new Set(this._urlsToCacheKeys.values());
      const deletedURLs = [];
      for (const request of currentlyCachedRequests) {
        if (!expectedCacheKeys.has(request.url)) {
          await cache.delete(request);
          deletedURLs.push(request.url);
        }
      }
      return { deletedURLs };
    });
  }
  getURLsToCacheKeys() {
    return this._urlsToCacheKeys;
  }
  getCachedURLs() {
    return [...this._urlsToCacheKeys.keys()];
  }
  getCacheKeyForURL(url) {
    const urlObject = new URL(url, location.href);
    return this._urlsToCacheKeys.get(urlObject.href);
  }
  getIntegrityForCacheKey(cacheKey) {
    return this._cacheKeysToIntegrities.get(cacheKey);
  }
  async matchPrecache(request) {
    const url = request instanceof Request ? request.url : request;
    const cacheKey = this.getCacheKeyForURL(url);
    if (cacheKey) {
      const cache = await self.caches.open(this.strategy.cacheName);
      return cache.match(cacheKey);
    }
    return void 0;
  }
  createHandlerBoundToURL(url) {
    const cacheKey = this.getCacheKeyForURL(url);
    if (!cacheKey) {
      throw new WorkboxError("non-precached-url", { url });
    }
    return (options) => {
      options.request = new Request(url);
      options.params = Object.assign({ cacheKey }, options.params);
      return this.strategy.handle(options);
    };
  }
}
let precacheController;
const getOrCreatePrecacheController = () => {
  if (!precacheController) {
    precacheController = new PrecacheController();
  }
  return precacheController;
};
try {
  self["workbox:routing:6.4.1"] && _();
} catch (e) {
}
const defaultMethod = "GET";
const normalizeHandler = (handler) => {
  if (handler && typeof handler === "object") {
    return handler;
  } else {
    return { handle: handler };
  }
};
class Route {
  constructor(match, handler, method = defaultMethod) {
    this.handler = normalizeHandler(handler);
    this.match = match;
    this.method = method;
  }
  setCatchHandler(handler) {
    this.catchHandler = normalizeHandler(handler);
  }
}
class RegExpRoute extends Route {
  constructor(regExp, handler, method) {
    const match = ({ url }) => {
      const result = regExp.exec(url.href);
      if (!result) {
        return;
      }
      if (url.origin !== location.origin && result.index !== 0) {
        return;
      }
      return result.slice(1);
    };
    super(match, handler, method);
  }
}
class Router {
  constructor() {
    this._routes = new Map();
    this._defaultHandlerMap = new Map();
  }
  get routes() {
    return this._routes;
  }
  addFetchListener() {
    self.addEventListener("fetch", (event) => {
      const { request } = event;
      const responsePromise = this.handleRequest({ request, event });
      if (responsePromise) {
        event.respondWith(responsePromise);
      }
    });
  }
  addCacheListener() {
    self.addEventListener("message", (event) => {
      if (event.data && event.data.type === "CACHE_URLS") {
        const { payload } = event.data;
        const requestPromises = Promise.all(payload.urlsToCache.map((entry) => {
          if (typeof entry === "string") {
            entry = [entry];
          }
          const request = new Request(...entry);
          return this.handleRequest({ request, event });
        }));
        event.waitUntil(requestPromises);
        if (event.ports && event.ports[0]) {
          void requestPromises.then(() => event.ports[0].postMessage(true));
        }
      }
    });
  }
  handleRequest({ request, event }) {
    const url = new URL(request.url, location.href);
    if (!url.protocol.startsWith("http")) {
      return;
    }
    const sameOrigin = url.origin === location.origin;
    const { params, route } = this.findMatchingRoute({
      event,
      request,
      sameOrigin,
      url
    });
    let handler = route && route.handler;
    const method = request.method;
    if (!handler && this._defaultHandlerMap.has(method)) {
      handler = this._defaultHandlerMap.get(method);
    }
    if (!handler) {
      return;
    }
    let responsePromise;
    try {
      responsePromise = handler.handle({ url, request, event, params });
    } catch (err) {
      responsePromise = Promise.reject(err);
    }
    const catchHandler = route && route.catchHandler;
    if (responsePromise instanceof Promise && (this._catchHandler || catchHandler)) {
      responsePromise = responsePromise.catch(async (err) => {
        if (catchHandler) {
          try {
            return await catchHandler.handle({ url, request, event, params });
          } catch (catchErr) {
            if (catchErr instanceof Error) {
              err = catchErr;
            }
          }
        }
        if (this._catchHandler) {
          return this._catchHandler.handle({ url, request, event });
        }
        throw err;
      });
    }
    return responsePromise;
  }
  findMatchingRoute({ url, sameOrigin, request, event }) {
    const routes = this._routes.get(request.method) || [];
    for (const route of routes) {
      let params;
      const matchResult = route.match({ url, sameOrigin, request, event });
      if (matchResult) {
        params = matchResult;
        if (Array.isArray(params) && params.length === 0) {
          params = void 0;
        } else if (matchResult.constructor === Object && Object.keys(matchResult).length === 0) {
          params = void 0;
        } else if (typeof matchResult === "boolean") {
          params = void 0;
        }
        return { route, params };
      }
    }
    return {};
  }
  setDefaultHandler(handler, method = defaultMethod) {
    this._defaultHandlerMap.set(method, normalizeHandler(handler));
  }
  setCatchHandler(handler) {
    this._catchHandler = normalizeHandler(handler);
  }
  registerRoute(route) {
    if (!this._routes.has(route.method)) {
      this._routes.set(route.method, []);
    }
    this._routes.get(route.method).push(route);
  }
  unregisterRoute(route) {
    if (!this._routes.has(route.method)) {
      throw new WorkboxError("unregister-route-but-not-found-with-method", {
        method: route.method
      });
    }
    const routeIndex = this._routes.get(route.method).indexOf(route);
    if (routeIndex > -1) {
      this._routes.get(route.method).splice(routeIndex, 1);
    } else {
      throw new WorkboxError("unregister-route-route-not-registered");
    }
  }
}
let defaultRouter;
const getOrCreateDefaultRouter = () => {
  if (!defaultRouter) {
    defaultRouter = new Router();
    defaultRouter.addFetchListener();
    defaultRouter.addCacheListener();
  }
  return defaultRouter;
};
function registerRoute(capture, handler, method) {
  let route;
  if (typeof capture === "string") {
    const captureUrl = new URL(capture, location.href);
    const matchCallback = ({ url }) => {
      return url.href === captureUrl.href;
    };
    route = new Route(matchCallback, handler, method);
  } else if (capture instanceof RegExp) {
    route = new RegExpRoute(capture, handler, method);
  } else if (typeof capture === "function") {
    route = new Route(capture, handler, method);
  } else if (capture instanceof Route) {
    route = capture;
  } else {
    throw new WorkboxError("unsupported-route-type", {
      moduleName: "workbox-routing",
      funcName: "registerRoute",
      paramName: "capture"
    });
  }
  const defaultRouter2 = getOrCreateDefaultRouter();
  defaultRouter2.registerRoute(route);
  return route;
}
function removeIgnoredSearchParams(urlObject, ignoreURLParametersMatching = []) {
  for (const paramName of [...urlObject.searchParams.keys()]) {
    if (ignoreURLParametersMatching.some((regExp) => regExp.test(paramName))) {
      urlObject.searchParams.delete(paramName);
    }
  }
  return urlObject;
}
function* generateURLVariations(url, { ignoreURLParametersMatching = [/^utm_/, /^fbclid$/], directoryIndex = "index.html", cleanURLs = true, urlManipulation } = {}) {
  const urlObject = new URL(url, location.href);
  urlObject.hash = "";
  yield urlObject.href;
  const urlWithoutIgnoredParams = removeIgnoredSearchParams(urlObject, ignoreURLParametersMatching);
  yield urlWithoutIgnoredParams.href;
  if (directoryIndex && urlWithoutIgnoredParams.pathname.endsWith("/")) {
    const directoryURL = new URL(urlWithoutIgnoredParams.href);
    directoryURL.pathname += directoryIndex;
    yield directoryURL.href;
  }
  if (cleanURLs) {
    const cleanURL = new URL(urlWithoutIgnoredParams.href);
    cleanURL.pathname += ".html";
    yield cleanURL.href;
  }
  if (urlManipulation) {
    const additionalURLs = urlManipulation({ url: urlObject });
    for (const urlToAttempt of additionalURLs) {
      yield urlToAttempt.href;
    }
  }
}
class PrecacheRoute extends Route {
  constructor(precacheController2, options) {
    const match = ({ request }) => {
      const urlsToCacheKeys = precacheController2.getURLsToCacheKeys();
      for (const possibleURL of generateURLVariations(request.url, options)) {
        const cacheKey = urlsToCacheKeys.get(possibleURL);
        if (cacheKey) {
          const integrity = precacheController2.getIntegrityForCacheKey(cacheKey);
          return { cacheKey, integrity };
        }
      }
      return;
    };
    super(match, precacheController2.strategy);
  }
}
function addRoute(options) {
  const precacheController2 = getOrCreatePrecacheController();
  const precacheRoute = new PrecacheRoute(precacheController2, options);
  registerRoute(precacheRoute);
}
function precache(entries) {
  const precacheController2 = getOrCreatePrecacheController();
  precacheController2.precache(entries);
}
function precacheAndRoute(entries, options) {
  precache(entries);
  addRoute(options);
}
const cacheOkAndOpaquePlugin = {
  cacheWillUpdate: async ({ response }) => {
    if (response.status === 200 || response.status === 0) {
      return response;
    }
    return null;
  }
};
class NetworkFirst extends Strategy {
  constructor(options = {}) {
    super(options);
    if (!this.plugins.some((p) => "cacheWillUpdate" in p)) {
      this.plugins.unshift(cacheOkAndOpaquePlugin);
    }
    this._networkTimeoutSeconds = options.networkTimeoutSeconds || 0;
  }
  async _handle(request, handler) {
    const logs = [];
    const promises = [];
    let timeoutId;
    if (this._networkTimeoutSeconds) {
      const { id, promise } = this._getTimeoutPromise({ request, logs, handler });
      timeoutId = id;
      promises.push(promise);
    }
    const networkPromise = this._getNetworkPromise({
      timeoutId,
      request,
      logs,
      handler
    });
    promises.push(networkPromise);
    const response = await handler.waitUntil((async () => {
      return await handler.waitUntil(Promise.race(promises)) || await networkPromise;
    })());
    if (!response) {
      throw new WorkboxError("no-response", { url: request.url });
    }
    return response;
  }
  _getTimeoutPromise({ request, logs, handler }) {
    let timeoutId;
    const timeoutPromise = new Promise((resolve) => {
      const onNetworkTimeout = async () => {
        resolve(await handler.cacheMatch(request));
      };
      timeoutId = setTimeout(onNetworkTimeout, this._networkTimeoutSeconds * 1e3);
    });
    return {
      promise: timeoutPromise,
      id: timeoutId
    };
  }
  async _getNetworkPromise({ timeoutId, request, logs, handler }) {
    let error;
    let response;
    try {
      response = await handler.fetchAndCachePut(request);
    } catch (fetchError) {
      if (fetchError instanceof Error) {
        error = fetchError;
      }
    }
    if (timeoutId) {
      clearTimeout(timeoutId);
    }
    if (error || !response) {
      response = await handler.cacheMatch(request);
    }
    return response;
  }
}
precacheAndRoute([{"revision":"6adcae76f0798dc97493b698dbaae85c","url":"assets/[...all].0d2e5810.js"},{"revision":"82f31a80528803f098da4d4facaf6e1a","url":"assets/[...all].17cf62d1.js"},{"revision":"bfef3aa279be23114d19fe5ec516340b","url":"assets/[...all].2ec35fc0.js"},{"revision":"849524367abfe28bc77f7f11c8a54e79","url":"assets/[...all].3408a0a3.js"},{"revision":"bc9065b5fae12d639a89494cf174ec03","url":"assets/[...all].4aac6cc0.js"},{"revision":"d9a89128eedf35c41abed3c08eebe07d","url":"assets/[...all].70c71f5a.js"},{"revision":"23863b5528a22a4cb2599690fc99ddb6","url":"assets/[...all].8b7b35d9.js"},{"revision":"131b968aad01a9bd088816fd0fd55a38","url":"assets/[...all].9172b434.js"},{"revision":"dbffede4cd31d4fab3434b28cee7fabd","url":"assets/[...all].c39ec37d.js"},{"revision":"33b2411a95bf7ebe212a66b7124dd397","url":"assets/[...all].cd05e41e.css"},{"revision":"b4f79342e27f3a04c99517780149aee6","url":"assets/[...all].d5bf12d7.js"},{"revision":"c2b068ccbd09f35056fc1292a52b1502","url":"assets/[...all].f676baee.js"},{"revision":"8d2079b1c5c02c23d7ed183fa6dd6929","url":"assets/app.157e7d5c.js"},{"revision":"35c2d73051269afda5191265fa5453a5","url":"assets/app.22689869.js"},{"revision":"a8fd4499f2a600df8847d575b3af0017","url":"assets/app.2d866bbd.css"},{"revision":"937cbaff1079d4091c3befb4512e265a","url":"assets/app.4dc05344.js"},{"revision":"387cda9c4c67f9d62b680fc5163b4eb2","url":"assets/app.5056b858.js"},{"revision":"abc8d33e692a453b6434fb461402dca0","url":"assets/app.50c40812.js"},{"revision":"969a4d0fba5d07d9f751b1898c6a646b","url":"assets/app.689926dc.js"},{"revision":"95ccdc3764f5116dadb9f6d004c89ccc","url":"assets/app.70cb7520.css"},{"revision":"5410325f33d8cd328e2d98615e9fc712","url":"assets/app.b47d5e80.js"},{"revision":"44e86c4eef8cfb48c3c0258ca0f26751","url":"assets/app.b57237d6.js"},{"revision":"40eb4eedfba8d90ab0a334fc705e7141","url":"assets/app.bb5c0c5a.js"},{"revision":"37665e1d105e871f61a161aa18da5da4","url":"assets/app.d09a40f6.js"},{"revision":"917bc39e31f9c07a99c26e2865141107","url":"assets/app.e011e054.js"},{"revision":"01e539b8e47468b9dab6a571b3a7b4f2","url":"assets/auth.01610c62.js"},{"revision":"35189f7db790b147805aa59ca7657ced","url":"assets/auth.125e9af2.js"},{"revision":"92025cf6221ba0760c313d0e66a16efd","url":"assets/auth.3beb8099.css"},{"revision":"662c4897eb8206cdcc48b608612f4a98","url":"assets/auth.5c218dc4.js"},{"revision":"3a2cdbf1d445a8d250f54f740971395a","url":"assets/auth.5c8cb547.js"},{"revision":"fc85f9f7f366f412f771134481c0a892","url":"assets/auth.651c8950.js"},{"revision":"9c39b80870514b7e850d1d99b7b990a6","url":"assets/auth.7cc0f6fb.js"},{"revision":"49e3dad0d2f436723b49030e40acf48f","url":"assets/auth.7dbfcfa4.css"},{"revision":"98ae3beb17f66561740e5b48c8b9b18e","url":"assets/auth.b1d908a4.js"},{"revision":"47528c77b8f01f6dddc26fda425ca16d","url":"assets/auth.b784ec6f.js"},{"revision":"e7df2bdb213f0f8977a4d844d43e6d84","url":"assets/auth.bbac7d1c.js"},{"revision":"4640190854c0be19a71357723efe553c","url":"assets/auth.c5e8beaf.js"},{"revision":"1943e5de512cba8d4da2793825f34769","url":"assets/auth.e57b52f4.js"},{"revision":"efdc6a8f238eb52bad95db87a24f3637","url":"assets/background.5bd78b71.js"},{"revision":"22e94b549b8e24065327949de18d1f21","url":"assets/commands.0234061a.js"},{"revision":"cba2d737cc743e93fcd5b059ddc25320","url":"assets/commands.14afe2ef.js"},{"revision":"95911fa70eba1741493437e487720d74","url":"assets/commands.275e6cd4.js"},{"revision":"7c121fdf902c983e0c5fc851f583cf76","url":"assets/commands.334226ff.js"},{"revision":"6ad1f8e4a653e40c54dbc93d1c4b5592","url":"assets/commands.6708939d.js"},{"revision":"671383a61cabee62ce32831e0c6ec2c2","url":"assets/commands.73f1c379.css"},{"revision":"53bc85ba465f8bebfc7fde98225f6c11","url":"assets/commands.9ff0bc83.css"},{"revision":"d4a8fa6c73bdb8320a55446301c12fc8","url":"assets/commands.a7a817fe.js"},{"revision":"3f7fdc3e46006cf64afe191b327c9e29","url":"assets/commands.bd630eb6.css"},{"revision":"e5a5d4ddbdaa5cba6c818b9142ff2297","url":"assets/commands.c138bebc.css"},{"revision":"6891a758bb565ea8ed8c84d125c4298f","url":"assets/commands.c41ce7c0.js"},{"revision":"e0c567371e525cc4721a9f2defe19943","url":"assets/commands.c6bc5d78.js"},{"revision":"524a7396cdf21cbcb8e37ffa54814e5e","url":"assets/commands.cafa957e.js"},{"revision":"844e830c4cf7aa572bf705afe448c23d","url":"assets/commands.d3991390.js"},{"revision":"3eb503cfdd91f22f317e41b86d48cd0c","url":"assets/commands.f4eb7718.js"},{"revision":"72e78068f7ff0648792ce2825b3b425b","url":"assets/has-nested-router-link.4fe8dab1.js"},{"revision":"123abe86b1e20eefcf479d6fd0932333","url":"assets/html2canvas.esm.0eae2bf4.js"},{"revision":"1d8a630723ff95558840824b19334f3e","url":"assets/index.1322291a.css"},{"revision":"f87eaefd2bd26bec9b42018b92337b5b","url":"assets/index.1584a8cb.js"},{"revision":"5bafbbd3339be3a54b3c2c646b3710c3","url":"assets/index.19a2e57f.js"},{"revision":"bf397f8a1801a6457fdaff44883e880b","url":"assets/index.1a838a4e.js"},{"revision":"8def2708fa643f4ff079665442c85fc0","url":"assets/index.248dc61a.js"},{"revision":"a8963bdf4ef1a717a442889f30b3a287","url":"assets/index.292089d9.js"},{"revision":"1b3c4479703594361f6ee3f5cd544815","url":"assets/index.295fbc93.css"},{"revision":"f44888d5a8ec04bae139bfc98e48473f","url":"assets/index.30eca897.js"},{"revision":"abb5fdd0383929f23e132586ba266b38","url":"assets/index.33831081.js"},{"revision":"2b99e05589243902cd2041314b3b3e41","url":"assets/index.34c37c3c.js"},{"revision":"36df8d69efae8e67f68b5fd084da7f43","url":"assets/index.34e397f1.js"},{"revision":"8b03c675a9b32fb0d80cc8405ba4191e","url":"assets/index.354dabdd.css"},{"revision":"2cb391ad1b3d5221a8b6a402c01d6cd0","url":"assets/index.3ef89e82.js"},{"revision":"a2629f9d16ca68ef567f55aafeef79a1","url":"assets/index.4087e787.css"},{"revision":"f30ae97dad7b99b5844236af87393dd0","url":"assets/index.41466198.js"},{"revision":"21682ad8fef87d69292d92c93fb4890c","url":"assets/index.41fc25d5.js"},{"revision":"b7d19bf37761c0cc6070ef460d8073df","url":"assets/index.4912b04c.js"},{"revision":"9f37391a0ab58dfca0fda5a64027f8a3","url":"assets/index.4d51ed56.js"},{"revision":"f2f698d8a92181b20ec1d4a2cbd98dc7","url":"assets/index.4f6a7bef.js"},{"revision":"71e2092e13bb55f680393beb6c268048","url":"assets/index.4fc608cd.js"},{"revision":"0aa51b1088ece772378a584e1f090175","url":"assets/index.516cc5d3.js"},{"revision":"193834d5715c144e85ba49553e9fcb37","url":"assets/index.51fb324e.js"},{"revision":"e4e4f0325641e862cedb51785f132f38","url":"assets/index.524653cc.js"},{"revision":"db2c1febc3eafc2e99e7567b47164bc4","url":"assets/index.591f3bfe.js"},{"revision":"2dd2fb72244a8d4b731853fae946589a","url":"assets/index.5d170f59.js"},{"revision":"0bdae69e42202e3b543d07883452ee78","url":"assets/index.5d881b98.js"},{"revision":"ccd4588bda6139e047388e0872cfebc2","url":"assets/index.6c851015.js"},{"revision":"5adce8961bd0ddddeed7f543a67a1e52","url":"assets/index.6cd1bdc5.js"},{"revision":"a3d08044b7a6dc56228a3f27c4aaea5d","url":"assets/index.7e30b23c.js"},{"revision":"01daea30da6d997285c8cdd56dc690c4","url":"assets/index.82e0aff8.js"},{"revision":"5016d7b5bf90868d392b64e8a00db17a","url":"assets/index.84a7f965.css"},{"revision":"b53be776dfb961738e42489cec41cbe3","url":"assets/index.86624c5a.js"},{"revision":"096544d8e950938d37105cfae5b35f61","url":"assets/index.89072ba7.js"},{"revision":"69c9e1a68baebbb308788af74a5df2db","url":"assets/index.95b05353.js"},{"revision":"73ccff532427d8d9c4a680c5af148ff0","url":"assets/index.9779eb17.css"},{"revision":"5fbcadd2ed403c227c5c6f2ba76a367c","url":"assets/index.9c43e512.js"},{"revision":"f0f173f38a216b47d997cad8f91d8b17","url":"assets/index.9ce64bc9.js"},{"revision":"a33ccdaab5d11fb24f5b6b9ba95aabd1","url":"assets/index.a66431e8.css"},{"revision":"73aab6c076e2c41d6be2a626f0a223a4","url":"assets/index.b4e3400d.css"},{"revision":"0b254d509e2788e1673e8b96d16b733d","url":"assets/index.c7341c5e.css"},{"revision":"550b3e19ae8eb82864b9ed79a5d16c78","url":"assets/index.cac0af1d.css"},{"revision":"f710ef73bbe98b37961c8185b612d194","url":"assets/index.cb725d5c.js"},{"revision":"90b2d8b5ab66ac08597400c6f6964c30","url":"assets/index.cb7a2565.js"},{"revision":"5651db0cd7878277f6cab1a40be82e65","url":"assets/index.cf02a220.js"},{"revision":"433d01717f04fa2d081d62961f608ee1","url":"assets/index.d0943639.js"},{"revision":"6cedaf06759927a480d988cc728b6286","url":"assets/index.d227a1e4.js"},{"revision":"b9edda2303484a610e24ca12ab99e5bc","url":"assets/index.d4bf3dc7.js"},{"revision":"15bbd2a9313be1a55936a0c785a2b719","url":"assets/index.d861acc9.css"},{"revision":"b90a3f3b89a209bbbe0fbee2d475f04f","url":"assets/index.d893df6f.js"},{"revision":"380fd375eb1187694a500055dcc2e284","url":"assets/index.e6d02c9b.css"},{"revision":"320c62fbc9e0781605824253151c2847","url":"assets/index.e78f651f.js"},{"revision":"bbaaca30f3b4718bb18d82f918662b86","url":"assets/index.e8e96972.js"},{"revision":"818d89120928b8de127e8e3278d32708","url":"assets/index.ec44bce5.css"},{"revision":"fb20b2a0a9083672568e2c7552f3cd90","url":"assets/index.efb5d479.css"},{"revision":"6629054bbdca00f3c09cc50fe8018056","url":"assets/index.es.105d732c.js"},{"revision":"c2d0ccb53bc19169e5d07efc6f07719f","url":"assets/index.es.22068a80.js"},{"revision":"967ae8437c7bb8e889e3258148e7fbab","url":"assets/index.es.3182dbb3.js"},{"revision":"e002403d0179ffb15deda4140991ef4b","url":"assets/index.es.6438b853.js"},{"revision":"1a4f6e08c0723831d66174d949b6c1ea","url":"assets/index.es.765fa7b4.js"},{"revision":"b7da9419e0f5095ab740ff362b4a80bc","url":"assets/index.es.fce22190.js"},{"revision":"d4afc0d03bdf05de0ff4cc7061c193f7","url":"assets/index.fd37ecff.css"},{"revision":"caf23f63f3bd281a85edd225d4e5989b","url":"assets/KanbanDropdown.01ad0287.js"},{"revision":"0e7bb362ab32bd51ffb893e8fd6a6af4","url":"assets/KanbanDropdown.1381bf32.js"},{"revision":"872c4d20d49058a165ce0f78bd4ba86d","url":"assets/KanbanDropdown.34128966.js"},{"revision":"09bdcb5a4897903fac2c76113cb4a108","url":"assets/KanbanDropdown.765d7434.js"},{"revision":"a733a7d29b949def3ff42d2cefdbeabc","url":"assets/KanbanDropdown.9886aff9.js"},{"revision":"df615f9efdbb331f36f049461a85a0be","url":"assets/KanbanDropdown.a5622d90.js"},{"revision":"87499e6a105e4ccfabbb375a514c3ea8","url":"assets/KanbanDropdown.b5e6e31c.js"},{"revision":"5c6c54b8556896a665725f727272275c","url":"assets/KanbanDropdown.ba4e8eaf.js"},{"revision":"180816371410abf240178df6a5481dcb","url":"assets/KanbanDropdown.c0d0cc8f.js"},{"revision":"7ac09bd5521375e27ccfc4e56df71eaa","url":"assets/KanbanDropdown.e4b6bc53.js"},{"revision":"e3e5a9ec8a494275ab1f729c51567dce","url":"assets/KanbanDropdown.ea902234.js"},{"revision":"b4c45f58534a53845b1951d89a56ddf8","url":"assets/login.4267dbb7.js"},{"revision":"dcf93b3b7be4ddaa40e56abb641033b4","url":"assets/login.590a5472.js"},{"revision":"b9b22269370cec07fb3ccc6ee3d13d4c","url":"assets/login.61560d3f.js"},{"revision":"695f54d60a35ea0264c2eb22d1973829","url":"assets/login.6ab3bcc8.js"},{"revision":"306ac02dcf6f1ede468ac6a44c6bacf7","url":"assets/login.969c79c2.js"},{"revision":"f76c61135404a5c9c32577d0f78a0eb4","url":"assets/login.9f5a0e59.js"},{"revision":"5f859b148d27a1326f0da00eb54670f8","url":"assets/login.acad7f6b.js"},{"revision":"d130051f8a7a5525fd556da4da941bf8","url":"assets/login.b952b423.js"},{"revision":"12a052556d59825c05a30b9da3a87ceb","url":"assets/login.c14b1e76.js"},{"revision":"d21a16a1fc3da8daaa88f132b19dbad0","url":"assets/login.e0eb5729.js"},{"revision":"96ba9f520b5b0952c7a6d45758e09d0f","url":"assets/login.eec379cc.css"},{"revision":"d0c02a0c569c6baed84674b46242e117","url":"assets/login.ef4ab935.js"},{"revision":"d415a3c92006cc560be3da0ffd9f18ba","url":"assets/masterService.179b61cf.js"},{"revision":"c6d3d9ccd6150a0479d7897adc3f83c6","url":"assets/masterService.1efb8c53.js"},{"revision":"453229ae14b67ad2c90bd92fb4a9fc74","url":"assets/masterService.297e33b7.js"},{"revision":"b683a5fea8efa7c5a40c90c8229d4b72","url":"assets/masterService.3aad9dac.js"},{"revision":"b4649bdff2b509df286526e605e09e5e","url":"assets/masterService.5f2c4e3c.js"},{"revision":"151fffb48a1f871cd95d962f74758efd","url":"assets/masterService.642860c6.js"},{"revision":"ebdf96bf2d04d6bd077f4691e62ac268","url":"assets/masterService.cf65685b.js"},{"revision":"1f41b5e7c2dedf207bbd51076d63d2ec","url":"assets/masterService.d887940f.js"},{"revision":"7b215dc6d48dad560f536ddb0684f4fe","url":"assets/masterService.dcf0adf2.js"},{"revision":"f0db2ffa9ae7ea905f5fcfa2a846f6a3","url":"assets/masterService.e0eebc43.js"},{"revision":"44411a17eb9db525a90d835b321c41dc","url":"assets/masterService.f79cdf3d.js"},{"revision":"0e2ad8d720cb15ea535d74cb515b8dd4","url":"assets/mesas.0c9dc222.css"},{"revision":"cca1a0b462f24ddeba6efede0c218b98","url":"assets/mesas.0e9ef2f1.js"},{"revision":"0ce8b9e628cf76e154783348be12bc8f","url":"assets/mesas.0eeedd13.js"},{"revision":"48181ae07fec30c2ddef196b07c55bb5","url":"assets/mesas.17fe5b5e.css"},{"revision":"8498f81a3f500fe67c26144bd69c3b31","url":"assets/mesas.240c58f3.js"},{"revision":"f1b2aafefbdbb5e1fbf07bc73dc8f001","url":"assets/mesas.289b63c8.js"},{"revision":"3fc7fb890fe1d3c77c9c061ba6f4dfef","url":"assets/mesas.29b36e75.js"},{"revision":"31992719734f024df3ce5bbc8b84df00","url":"assets/mesas.34452eec.js"},{"revision":"7d3898444b67dbecb08023f6c0db99fb","url":"assets/mesas.3dc4b5ff.js"},{"revision":"a63eda9e0bda2970134e5978c1666f2f","url":"assets/mesas.411d4e10.css"},{"revision":"3b6c3e855e9084e0152fff77b99960a7","url":"assets/mesas.5a54e66d.js"},{"revision":"aaeb8213e159e3e2b4802afa586f2735","url":"assets/mesas.73bbb649.js"},{"revision":"9961e0eaa41b51dbb503a437147d6a6e","url":"assets/mesas.788333a1.js"},{"revision":"6a1376ef584681b9767850490bcaceae","url":"assets/mesas.82089356.js"},{"revision":"21906f6e0a81a8d3266d537546233667","url":"assets/mesas.885bc9ee.js"},{"revision":"c57f0451763957b51ffd3b5d99ab060d","url":"assets/mesas.8b109f40.css"},{"revision":"89668b0c329b2e0ada209b3fd32234a1","url":"assets/mesas.8b55237e.js"},{"revision":"5f9d1f725e0e84cd51101d39a1b33c59","url":"assets/mesas.90d25141.js"},{"revision":"8285c684f022d50f32bd590c3b2eadf2","url":"assets/mesas.95621fc8.js"},{"revision":"4f32dec4244b896433f1d22a86131895","url":"assets/mesas.9bd22492.js"},{"revision":"e1c33a62264991f34fc53e2f506b8f67","url":"assets/mesas.bf3e9023.js"},{"revision":"eb12df3af1b78926be7adf36988a41c6","url":"assets/mesas.c09ff87e.css"},{"revision":"55b1b853a89b50ff26e71729f114322c","url":"assets/mesas.cb93f666.css"},{"revision":"e2699bb29c6984baf44266b6ecaea9ed","url":"assets/mesas.d1b121c6.js"},{"revision":"b8efc08f35b4be03468ac1aab40a71f5","url":"assets/mesas.d8609bbd.js"},{"revision":"96e55ba0371f239d3c79d5c1415919a8","url":"assets/mesas.ee86d0b2.js"},{"revision":"81c0e4380c2b16ba1b94f90d24cc9a80","url":"assets/mesas.fd2037cb.js"},{"revision":"0d2ac1984f8e280ad364910bfc21b3a3","url":"assets/mesas.ffac6cef.js"},{"revision":"dbecc4b6d4a3e8a33b5b7bb7af424e9f","url":"assets/mesaService.03478fb3.js"},{"revision":"df00a486d50c0dcbed41de3d9510aff9","url":"assets/mesaService.21a7d313.js"},{"revision":"f3d6616e714b541adcf5eafeb4506744","url":"assets/mesaService.27a56e32.js"},{"revision":"af8efa62886aad29c44069b3ec8d7dbb","url":"assets/mesaService.4fc9e848.js"},{"revision":"7a4895a18874d886bd96af832fef5269","url":"assets/mesaService.51000732.js"},{"revision":"88921a5bef094b0a75b6a8c849562c9f","url":"assets/mesaService.625fb5f4.js"},{"revision":"291d605a4f1e023449c870b85d3c905e","url":"assets/mesaService.6da053b4.js"},{"revision":"4e003266a86c70986a733276c6fcddf2","url":"assets/mesaService.8e343c4f.js"},{"revision":"bad718342ce863419ed7eacfd9c206a5","url":"assets/mesaService.a6ec9502.js"},{"revision":"10fbf59b3e20129dd827a38a3a2afb54","url":"assets/mesaService.b0a07ad0.js"},{"revision":"8061f11c5dab67ec3371782bbd6c5bbe","url":"assets/mesaService.f0042f10.js"},{"revision":"8e6dffb5c7b5ad9c2648c75364d2077f","url":"assets/mozo.10b24dec.js"},{"revision":"2b4c1f592b2c615a995e22bb84c6777f","url":"assets/mozo.19855efc.css"},{"revision":"e848d18760ae8cea87bf624dd44da5c9","url":"assets/mozo.3491b1f9.js"},{"revision":"c24f5dc68566917de788aecdb0cd40d8","url":"assets/mozo.5f58d83c.js"},{"revision":"27e20af117ff35622e0b5d391afe9655","url":"assets/mozo.7dea9877.js"},{"revision":"e9da02f07ffdfbc823e65309ebc0f811","url":"assets/mozo.8c187282.js"},{"revision":"07e0f2552712f1879f9e066532f25f4f","url":"assets/mozo.99700952.js"},{"revision":"ac1bbf1eddef565d62ea13a4e1729e76","url":"assets/mozo.c6cf2f13.css"},{"revision":"8d935604b00f989a1684a06a25038b16","url":"assets/mozo.d399dd7e.js"},{"revision":"755479318857b3fc4f9bcfa99517216b","url":"assets/mozo.d594d72b.js"},{"revision":"1ad3f46c53b3ab7310567438b7239032","url":"assets/mozo.e6ba9bd7.css"},{"revision":"48582e67471801da5440a98a1ce4a3b5","url":"assets/mozo.e9600c27.js"},{"revision":"b51bf6e734110efe211abfd7d0dc065c","url":"assets/multiselect.38dd90cc.js"},{"revision":"e822ed3c594553072004c511426e6233","url":"assets/multiselect.65b9f05a.js"},{"revision":"76b9c41f1cc71e32b6ceacc402401101","url":"assets/multiselect.796f84fb.js"},{"revision":"e53c7e0cd3a5b2326d8d97c723118818","url":"assets/multiselect.9736f255.js"},{"revision":"310789f959bc7b081b37b4d0c438885d","url":"assets/multiselect.b62544b5.js"},{"revision":"b7e0e6e2b5fed33c46b199d146c35363","url":"assets/multiselect.b6db6c00.js"},{"revision":"fd87351a714e0c28080eb9a10dab9b6d","url":"assets/orders.1a1e42dc.js"},{"revision":"6c354d3343e779fcf545da16d455aac0","url":"assets/orders.2b72a5d6.js"},{"revision":"1bb681ec95e6bfc20fe75604800cabad","url":"assets/orders.57d563ac.js"},{"revision":"b971671d98436348915bdc306a2d3e41","url":"assets/orders.5fcf1b12.js"},{"revision":"6bf1245e75a394582ecc26f563096cb2","url":"assets/orders.6a4294a5.js"},{"revision":"d7be992fa084e87dff184d721234b4ce","url":"assets/orders.76f991a4.css"},{"revision":"295a51b7c52d1400bf6a2cd4c3fd9bdf","url":"assets/orders.8b4ff1da.js"},{"revision":"c06b83a5634a85cf64f874fd9ee819b2","url":"assets/orders.8b7baeed.js"},{"revision":"552d47f4c06103c4e8186d8fdcf4f206","url":"assets/orders.b8d16e60.js"},{"revision":"a9ea3e47b492f40edd5d74c23912104e","url":"assets/orders.c85ca1b7.css"},{"revision":"9134f39ef330eb9d5d633150913fe07e","url":"assets/orders.d107036b.js"},{"revision":"a1284913d5404fda0f96bba0ba2560a8","url":"assets/orders.f1b73f73.js"},{"revision":"95d95af3f6eca4b5a97bbfa1042f9d23","url":"assets/orders.f1f20ba9.js"},{"revision":"ca051e4b1897ce451867683b8bc731b0","url":"assets/pos.13a5283e.js"},{"revision":"db9dc154f4943cde57b36b49e1047c96","url":"assets/pos.26d22cad.js"},{"revision":"fa71a12e005cf954fbb991a773679375","url":"assets/pos.29a677ab.js"},{"revision":"d3061f12924fe8a9477d989cf62f9894","url":"assets/pos.2cbc42c7.js"},{"revision":"81b2ffa5b4d4152166a41a90f14fbbb9","url":"assets/pos.3c11e7c9.js"},{"revision":"050ba13f7f3e945113f3f65417f96fef","url":"assets/pos.46967246.js"},{"revision":"ff63ab5a49ac05285b313fab5a02f30c","url":"assets/pos.6559ed48.js"},{"revision":"d55f8bf99b84032a9bdebe10e83fd1c2","url":"assets/pos.66e0bc5e.js"},{"revision":"71a14892ddb8621582e3a2472303bca7","url":"assets/pos.6775a4d6.js"},{"revision":"f6a9882aa7a1d1719032cac71cfc81b5","url":"assets/pos.699e9668.js"},{"revision":"99260dc901bff7ab8f00e1c91b62b824","url":"assets/pos.787eab8d.js"},{"revision":"804075d15491a66370915ac8a5460755","url":"assets/pos.7f63bb43.js"},{"revision":"1857ef806ca8d9cea2a2ff83aa4fa392","url":"assets/pos.9ca6431d.js"},{"revision":"34a1b27d23c0ced212f3bc08b11463f9","url":"assets/pos.a652753b.js"},{"revision":"5c9c00d8e5bcec9436eace267f39db97","url":"assets/pos.b4cfebd0.js"},{"revision":"1d05fc10665093e5e30b286c2062b059","url":"assets/pos.b5271657.js"},{"revision":"8374c6ec4e7047ca744d0902203bdd60","url":"assets/pos.db1bf117.js"},{"revision":"f511733a67ae3b5a264c0df4b53079aa","url":"assets/prices.0b1011ed.js"},{"revision":"4b791081fafc9b95fb69e8972798b182","url":"assets/prices.17138680.js"},{"revision":"8653c2f67b78240b28a4e5172062ea8d","url":"assets/prices.1a494096.js"},{"revision":"4f2de68d03c3ed2e18e1f8c073e04466","url":"assets/prices.454d4dd8.css"},{"revision":"ec9fe78049b6239b508a386ee0da823c","url":"assets/prices.4938cfe9.js"},{"revision":"a4379961a71966354a4b1ddfffbfa09e","url":"assets/prices.5d1c25b4.js"},{"revision":"e4a23e7d045882bf97f43da88df8ced4","url":"assets/prices.61532670.js"},{"revision":"53afa77e4d62e88524fb9bdae9153907","url":"assets/prices.6b9da9a9.js"},{"revision":"d15ead03654f518a9bfa574e6291eacb","url":"assets/prices.a820c97f.js"},{"revision":"fbb9c3739f486f520cb5fae055229238","url":"assets/prices.c7a385ec.js"},{"revision":"2c315d4870f4b159a2256d13efc96ad0","url":"assets/prices.ce23ca5c.js"},{"revision":"e7d8b3de895c9c46cc72a68c253976ec","url":"assets/prices.f8ad801f.js"},{"revision":"f17d31eb8f49e10d9d7a3b1d8f684521","url":"assets/purify.es.82af1ade.js"},{"revision":"84189282b023779c67c7c4485ae6ca70","url":"assets/sidebarLayoutState.037b92f3.js"},{"revision":"9b2431b2de192a66ab6d61db98582175","url":"assets/sidebarLayoutState.08f3edf0.js"},{"revision":"db086011d0977c660509983e29dbc6fa","url":"assets/sidebarLayoutState.386da090.js"},{"revision":"b81781f4475ca8b0d8fb20098fc47493","url":"assets/sidebarLayoutState.41819060.js"},{"revision":"d50d946d459c3635ddca9405d4d7a6de","url":"assets/sidebarLayoutState.bc85911b.js"},{"revision":"5d586142c71e0b26594fe23c2266fbcf","url":"assets/sidebarLayoutState.c465df13.js"},{"revision":"73169461688c5a4308b9454024ffc2d5","url":"assets/signup.0acfc957.js"},{"revision":"097c0d015276c91f8172673236619c9e","url":"assets/signup.2f0623ab.js"},{"revision":"e641a5ebd4a1578ed693a66902838551","url":"assets/signup.4c86cefe.js"},{"revision":"d07948a6878ff2f29a25eef9a5fc4b85","url":"assets/signup.55830f94.js"},{"revision":"594e8515e6f351151553cf2df7016ac4","url":"assets/signup.64f4a53e.js"},{"revision":"fad1c5e02efbf71830c77aa52356e2e5","url":"assets/signup.873e4b61.js"},{"revision":"a52422c21a39c0c1264911eb0e99fdda","url":"assets/signup.b1be2cac.js"},{"revision":"8aed840ba87f741ce66bf90c0d0c1639","url":"assets/signup.b2bed3b6.js"},{"revision":"89b9b77cd308275ce25367002947e26f","url":"assets/signup.b7956c5f.js"},{"revision":"4cd484ce0d08031814ea3b6d575ee67d","url":"assets/signup.dcdda218.js"},{"revision":"8eebbe11fd262321698cc00478acc4a2","url":"assets/signup.f4a7bfd8.js"},{"revision":"8f2ebf23113f9b0bc24203d1ee71a26c","url":"assets/slider.074e2ff8.js"},{"revision":"fe762f40737913f0e83118f2588ae3b8","url":"assets/slider.09c72948.js"},{"revision":"df2b89b799bb23c4b4d00be19a930b8a","url":"assets/slider.0eeedf40.js"},{"revision":"c28fe543b89c29c66e668966a37770ed","url":"assets/slider.9f084a13.js"},{"revision":"568c7664170a3fda1164361c013827b1","url":"assets/slider.c17b06ac.js"},{"revision":"e523f6d71907d337f1f5a33f8c168036","url":"assets/slider.dbf5407b.js"},{"revision":"a919e07cc439a2e336d14a0ae71f119a","url":"assets/tooltip.24128ff9.js"},{"revision":"8bcba5678ab9cc3868fddd73660ae67b","url":"assets/useViaPlaceholderError.be9c33a4.js"},{"revision":"e597b29002ca510b8cddcd9cf1c5b444","url":"assets/VAvatar.1a51a87f.js"},{"revision":"f9112fae9c5c7af91e1136b476a904e2","url":"assets/VAvatar.367064ef.js"},{"revision":"91ec814fa849b10f47821f43498fcf20","url":"assets/VAvatar.4c1ed57a.js"},{"revision":"6da9339ba03fc90bb82d233d8ff7702f","url":"assets/VAvatar.819b042b.js"},{"revision":"87fdb844e17add854321f9b55c3c5c5b","url":"assets/VAvatar.864329ab.js"},{"revision":"a854c10b3f839c7c2e414a3a44edcdcb","url":"assets/VAvatar.b904ec8e.js"},{"revision":"cb59023dd395e24c983fc50b553a838a","url":"assets/VButtons.5078402e.js"},{"revision":"209c0d79c3f5ea695dd9de4f0f8c13ae","url":"assets/VButtons.79021ab7.js"},{"revision":"e8745d7610c453e2e2e3a7846045c948","url":"assets/VButtons.80a94f5a.js"},{"revision":"23f4080a8cbc82c4444ad15d96b58794","url":"assets/VButtons.b45a77f4.js"},{"revision":"02d8f5cd71e5d2212fcd591b908fe58b","url":"assets/VControl.2b5f0903.js"},{"revision":"b089ab7319b0b585c6255f5c46a0ffa5","url":"assets/VControl.36f1f0b8.js"},{"revision":"ab0823ca937aa366c6cca5d06201bf96","url":"assets/VControl.55e3dd24.js"},{"revision":"5d4dbc6b708ccd1996284e10f1c8e36d","url":"assets/VControl.58ce19cc.js"},{"revision":"553b96604a1bb3ec3ef6d7b517942fe1","url":"assets/VControl.66ed690d.css"},{"revision":"e633a00f5614d92b02019b4515e11a63","url":"assets/VControl.68d2089b.js"},{"revision":"58c18046737c990c23a491004b163c73","url":"assets/VControl.87ec9af6.js"},{"revision":"b5e30bc2486c171ae2d3beca32c88fe9","url":"assets/VControl.901448b0.js"},{"revision":"c0ac2e4d029d24700d0422cad60a242d","url":"assets/VControl.abb75441.js"},{"revision":"4277140738a8dbf120e0d2c886b24fac","url":"assets/VControl.c1dfb3fe.js"},{"revision":"86b52273f9032a5fad8b695e8f2b718d","url":"assets/VControl.dfb4cb15.js"},{"revision":"2c3f7a1a988f578145389dcc716fb1aa","url":"assets/VControl.f47dfef8.js"},{"revision":"b441da8555345bfafce2c53d158f532a","url":"assets/VDropdown.107a6678.js"},{"revision":"d0734fdca85ef37c38e9720a946c919e","url":"assets/VDropdown.247e8893.js"},{"revision":"4cc05bffab009afe66e4a0528a5477c0","url":"assets/VDropdown.2573bd4c.js"},{"revision":"193ab921683b4fc3d08d326676e12727","url":"assets/VDropdown.31f1f578.js"},{"revision":"26e31d7b05962ddc4519733081cc5019","url":"assets/VDropdown.8e1ab7d8.js"},{"revision":"a3148ec1d61c0eecbe4c50b7337fee1b","url":"assets/VDropdown.a7b68ccd.css"},{"revision":"ccaab1e8a4d8bd6b2a579d4bae5bf116","url":"assets/VDropdown.b2180179.js"},{"revision":"1e2c6c08a548a9f839db8a3124bbb633","url":"assets/VDropdown.b2341ff3.js"},{"revision":"93c19b7db617cb7b53b6df07fdc1a59b","url":"assets/VDropdown.cef1fa42.js"},{"revision":"fed5ae81266c56dda8105faa1d0fc186","url":"assets/VDropdown.e70a3879.js"},{"revision":"85261ffb2cb9794fdd10aa96661c6271","url":"assets/VDropdown.e8d260ba.js"},{"revision":"fb747db04a746423c2d7226a9e97805c","url":"assets/VDropdown.ee04f776.js"},{"revision":"1238f8178c835db29fc51cbbca6bdc7d","url":"assets/vendor.48417147.js"},{"revision":"ed1200fd5ed399ab41f3a8c6e9fa7818","url":"assets/vendor.4b7a2e6c.js"},{"revision":"2f91ee186054752579267f62372ecf06","url":"assets/vendor.55da2271.js"},{"revision":"18a5a369f43b89d6c2d867ec5017e978","url":"assets/vendor.7db31232.js"},{"revision":"64c7f849e5c5fcf961d72edac1ef0b81","url":"assets/vendor.d4ec66ae.js"},{"revision":"cec261bddf4e1c114b506d02b3044407","url":"assets/vendor.dd48a589.js"},{"revision":"0b65a1559d9841d75c913f54f06f26d8","url":"assets/VField.91553a3a.js"},{"revision":"26d73c15de628b567aa49274cf2afee0","url":"assets/VField.ae64c3d0.js"},{"revision":"bd7ebd709b195a1be04d6e2a368fe027","url":"assets/VField.afaabb17.js"},{"revision":"c9671983b63d97a6506eccf2e092db27","url":"assets/VField.b2ff2f2a.js"},{"revision":"f9c088db9e44135cda3950e24839935b","url":"assets/VField.b3125b15.js"},{"revision":"aaa9fe57d61315be9c3d2058f385e4ef","url":"assets/VField.e1db8518.js"},{"revision":"05fe4280ad7b8314916d3559e132556d","url":"assets/VIcon.1e784500.js"},{"revision":"e372ea38522472f823419d1ec6aa7a28","url":"assets/VIcon.64a302a7.js"},{"revision":"3fb708c42026417bc6e95588b773c15b","url":"assets/VIcon.a165e41f.js"},{"revision":"60f473158e7dcc40ed18d1c290c67499","url":"assets/VIcon.a5f7aa28.js"},{"revision":"10efbf63e95da68cbb431f8fe4f07362","url":"assets/VIcon.c393c789.js"},{"revision":"6ccecaba44d7cfe56227748d6353e64a","url":"assets/VIcon.dd77c06e.js"},{"revision":"623982c7de982a1e89c0c1cae958a15e","url":"assets/VIconButton.0ec33bf6.js"},{"revision":"9bfda185489bf1638c5207f2b000c287","url":"assets/VIconButton.1ffac9fc.js"},{"revision":"c13eb2c6f461873ba69f432d072ad6a6","url":"assets/VIconButton.30e2f6de.js"},{"revision":"25cb43e8ef9c4f8852d877624a537329","url":"assets/VIconButton.4aa07a0a.js"},{"revision":"d22aa82e464a4ed1eaf7cddf71bf99f6","url":"assets/VIconButton.7ea0424d.js"},{"revision":"36787643e88386ead0870a4da50f183e","url":"assets/VIconButton.88612794.js"},{"revision":"c6b16495f6ac523e11616d6318810655","url":"assets/VIconButton.9fc03028.js"},{"revision":"842434111bed8db06112695ba572f33f","url":"assets/VIconButton.a39b0b9b.js"},{"revision":"da695dc625e526a26a54286624177c11","url":"assets/VIconButton.b75a74f0.js"},{"revision":"02c465093f2eaa00c4707013732a8048","url":"assets/VIconButton.c030365a.js"},{"revision":"1954e0b6599251de2ceb332905af0c69","url":"assets/VIconButton.e21b1740.js"},{"revision":"737298a2c45ee9a73423e0c1721bffb6","url":"assets/VLoader.05ecf071.js"},{"revision":"285c15a7fb57610c575960e7b6ec4edc","url":"assets/VLoader.08d53c1d.js"},{"revision":"e96b93a78dbfb0c8fd2f91eabe535d4c","url":"assets/VLoader.3dcc431b.js"},{"revision":"ec33e59ab278b8a3b5424e84f033b5ec","url":"assets/VLoader.7fe927ed.js"},{"revision":"0bc0e72038060080781bf3e3360ff48b","url":"assets/VLoader.b763de9d.js"},{"revision":"325e9889e26d552d30b82150762a86c9","url":"assets/VLoader.f5ba2b30.js"},{"revision":"653cb7f1a7a81455fe66893e08f8b4f3","url":"assets/VModal.12b66599.js"},{"revision":"66dc6c80007f1bfbd544167dd33d95ef","url":"assets/VModal.2ef11452.js"},{"revision":"362c6b5ed1b7174b399f4baf6b2f9d35","url":"assets/VModal.5c485a91.js"},{"revision":"5e8076ecf7eaeb48bd5e9278e260068e","url":"assets/VModal.af15beb6.js"},{"revision":"2efa864dca2c26bfcaa1a4826eb9504f","url":"assets/VModal.d8de09e0.css"},{"revision":"c63490a88f67b2428b54bf2af7dc0db7","url":"assets/VModal.dbbfe8dd.js"},{"revision":"e2ddf42ff6305e3e40dcc6e30b59cfa3","url":"assets/VModal.eb1ab153.js"},{"revision":"6dbfebd2a94b683b619d2299d23d261c","url":"assets/VPlaceloadText.363df2d8.js"},{"revision":"318e5e372506e6693445d76404fa3311","url":"assets/VPlaceloadText.43f5965f.js"},{"revision":"93b206e1c3e84049521c620c06cbb5e1","url":"assets/VPlaceloadText.58d57dbf.js"},{"revision":"896b717a25295c6be7036f73ea61952c","url":"assets/VPlaceloadText.642d31d3.js"},{"revision":"53665782b641714a569f71714226e430","url":"assets/VPlaceloadText.6fe1b7bc.js"},{"revision":"a4a820c7bdcb3ef618877085e73fbab6","url":"assets/VPlaceloadText.98618876.js"},{"revision":"40b4784402057e2b35f0b784771bc039","url":"assets/VPlaceloadText.a3660a8f.js"},{"revision":"f71c8443b764a7d872a0f60796e14e27","url":"assets/VPlaceloadText.abeaf7d8.js"},{"revision":"eab4036d19d0b2478ae894d685ae1437","url":"assets/VPlaceloadText.b27a1d27.js"},{"revision":"e7d12ac7bec303511cc0eff6b4310a23","url":"assets/VPlaceloadText.eb0f06c9.js"},{"revision":"cdd5b6c2fc998d0d0d927ea454061132","url":"assets/VPlaceloadText.f4b5fc2d.js"},{"revision":"96cc2265d553efef29a60a5e4c657e76","url":"assets/VPlaceloadWrap.08632626.js"},{"revision":"1b41beffa84112270ad32e61e9673325","url":"assets/VPlaceloadWrap.53cbd965.js"},{"revision":"bebaed3d551b7a2941acec4c61e0d52e","url":"assets/VPlaceloadWrap.6c1d3fee.js"},{"revision":"fb26cef6cb2a05c3859574179bb3f39d","url":"assets/VPlaceloadWrap.7b63bb61.js"},{"revision":"eed321f7e3589ec1ceb84db41ae195dc","url":"assets/VPlaceloadWrap.82eb1f61.js"},{"revision":"5e5576af886a411ac29a83b9eaa4f633","url":"assets/VPlaceloadWrap.914e2783.js"},{"revision":"c78527488d65d2f694a3c8cda7aa5e92","url":"assets/VPlaceloadWrap.91821902.js"},{"revision":"cb66bda51abd633358cb40597e4c871b","url":"assets/VPlaceloadWrap.9e9678b9.js"},{"revision":"99359a4dc5431ff30d635e09e33adb0b","url":"assets/VPlaceloadWrap.a21e0bfa.js"},{"revision":"78a707b365a9bdace7783699c6620415","url":"assets/VPlaceloadWrap.a9cb2b21.js"},{"revision":"e0bcbec211ade5a8324e4dc58de6c1f6","url":"assets/VPlaceloadWrap.f6b4da16.js"},{"revision":"9d4e9937ffc09fa388d1dc3db7c8b472","url":"assets/vue-tippy.esm-bundler.081b0a98.js"},{"revision":"c662ae4fcc2c3e1ccf7c58fbc74e7c62","url":"assets/vue-tippy.esm-bundler.13bfe509.js"},{"revision":"db99ec5320d9ab1e83bd9547ea50345f","url":"assets/vue-tippy.esm-bundler.7ba00f27.js"},{"revision":"05ba013a80440c6b89d3e3db45dde3c2","url":"assets/vue-tippy.esm-bundler.897768fe.js"},{"revision":"31281d5f6e6403930be4d2d6c2cf718d","url":"assets/vue-tippy.esm-bundler.bdbfb7ba.js"},{"revision":"33b22980aacc4b0071b19c9627a51f64","url":"assets/vue-tippy.esm-bundler.eece087d.js"},{"revision":"690b4e440a24af82ba04710f56fb15d0","url":"index.html"},{"revision":"d10bf770c774886de92bf247c64e674b","url":"vendors/font-awesome-v5.css"},{"revision":"4f9e480835662341f352d3262e6baff2","url":"vendors/line-icons-pro.css"},{"revision":"46c80fa08da7df61e0300ce3697ac93f","url":"vendors/loader.js"},{"revision":"eda6bc09b35dd6194bd89102886f68f2","url":"vendors/prism-coldark-cold.css"},{"revision":"aa8095a9d6bd9466bfdee6a9e0fb143e","url":"favicon.svg"},{"revision":"2608995d3ce047aed1b4f12314b971e6","url":"favicon.ico"},{"revision":"cd9cd94aaa699e0a16e692b6bb16f672","url":"robots.txt"},{"revision":"b1fc7ba21cbe0c252ddf4e374dff5bcf","url":"apple-touch-icon.png"},{"revision":"598ac9f6ba4777c6a0839a61f484cc95","url":"pwa-192x192.png"},{"revision":"fdde4a327d6c825b405236efbb8da6e3","url":"pwa-512x512.png"},{"revision":"c075bfe56460ed58a568c87523377d1d","url":"manifest.webmanifest"}]);
registerRoute(({ url }) => url.href.startsWith("https"), new NetworkFirst());
